#include <chrono>
#include <cstddef>
#include <dpp/dpp.h>
#include <future>
#include <iomanip>
#include <iostream>
#include <regex>
#include <sstream>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <string>

#include "commands/abstract.h"
#include "commands/help.h"
#include "commands/join.h"
#include "commands/jump.h"
#include "commands/leave.h"
#include "commands/kill.h"
#include "commands/lyrics.h"
#include "commands/move.h"
#include "commands/next.h"
#include "commands/pause.h"
#include "commands/play.h"
#include "commands/playnext.h"
#include "commands/queue.h"
#include "commands/remove.h"
#include "commands/resume.h"
#include "commands/queue-search-jump.h"

#include "commands/shortcutadd.h"
#include "commands/shortcutremove.h"
#include "entities/shortcut.h"
#include "utils/config.h"
#include "utils/db-migration.h"
#include "utils/db.h"
#include "utils/general.h"

void sendErrorEmbed(dpp::cluster &bot, dpp::message_create_t event, std::string title, std::string description = std::string(), std::string mediaUri = std::string());
void sendEmbed(dpp::cluster &bot, uint32_t color, dpp::message_create_t event, std::string title, std::string description, std::string mediaUri);
bool isPlayCommandRunning(dpp::snowflake guildId, std::map<dpp::snowflake, std::shared_future<void>> playCommandFutures);

int main(int argc, char* argv[])
{
    if (argc > 1)
        ConfigUtil::initConfig(argv[1]);
    else
        ConfigUtil::initConfig();

    DbMigrationUtil::migrate();

    uint64_t intents = dpp::i_default_intents | dpp::i_message_content;
    dpp::cluster bot(ConfigUtil::get("bot_token", true), intents);

    std::map<dpp::snowflake, std::shared_future<void>> playCommandFutures;

    //Start low priority queue thread
    std::queue<std::function<void()>> lowPriorityQueue;
    std::thread lowPriorityThread([&lowPriorityQueue]
                                  {
        while (true)
        {
            if (lowPriorityQueue.empty())
            {
                std::this_thread::sleep_for(std::chrono::milliseconds(500));
                continue;
            }

            std::function<void()> job = lowPriorityQueue.front();

            job();
            lowPriorityQueue.pop();
        } });
    lowPriorityThread.detach();

    //Create vector of all available commands
    std::vector<std::shared_ptr<AbstractCommand>> commands;
    PlayCommand playCommand = PlayCommand(bot);
    JumpCommand jumpCommand = JumpCommand(bot, playCommand, playCommandFutures);
    commands.push_back(std::shared_ptr<AbstractCommand>(new HelpCommand(bot, commands)));
    commands.push_back(std::shared_ptr<AbstractCommand>(new JoinCommand(bot)));
    commands.push_back(std::shared_ptr<AbstractCommand>(new QueueCommand(bot)));
    commands.push_back(std::shared_ptr<AbstractCommand>(&playCommand));
    commands.push_back(std::shared_ptr<AbstractCommand>(new RemoveCommand(bot)));
    commands.push_back(std::shared_ptr<AbstractCommand>(new LeaveCommand(bot, playCommand)));
    commands.push_back(std::shared_ptr<AbstractCommand>(new KillCommand(bot, playCommand)));
    commands.push_back(std::shared_ptr<AbstractCommand>(new PauseCommand(bot)));
    commands.push_back(std::shared_ptr<AbstractCommand>(&jumpCommand));
    commands.push_back(std::shared_ptr<AbstractCommand>(new NextCommand(bot, playCommand, playCommandFutures)));
    commands.push_back(std::shared_ptr<AbstractCommand>(new PlayNextCommand(bot, playCommand)));
    commands.push_back(std::shared_ptr<AbstractCommand>(new MoveCommand(bot)));
    commands.push_back(std::shared_ptr<AbstractCommand>(new ResumeCommand(bot, playCommand, playCommandFutures)));
    commands.push_back(std::shared_ptr<AbstractCommand>(new ShortcutAddCommand(bot)));
    commands.push_back(std::shared_ptr<AbstractCommand>(new ShortcutRemoveCommand(bot)));
    commands.push_back(std::shared_ptr<AbstractCommand>(new QueueSearchJumpCommand(bot, playCommand, jumpCommand)));

    std::string accessToken = ConfigUtil::get("genius_access_token");
    if (!accessToken.empty() && accessToken != "GENIUS_ACCESS_TOKEN")
    {
        commands.push_back(std::shared_ptr<AbstractCommand>(new LyricsCommand(bot)));
    }

    //Log to console on bot ready
    bot.on_ready([&bot](const auto &event)
                 {
                     std::cout << "Bot ready. Username: " << bot.me.username << "\n";
    bot.set_presence(dpp::presence(dpp::presence_status::ps_online, dpp::activity_type::at_listening, "L.O.U.T.R")); });

    bot.on_message_create([&bot, &commands, &playCommand, &lowPriorityQueue, &playCommandFutures](const dpp::message_create_t &event) {
                              std::string command = event.msg.content;
                              for (int i = 0; i < commands.size(); i++)
                              {
                                  //Loop through regex expressions and check if one matches
                                  std::vector<std::string> regex = commands[i]->getRegex();
                                  for (int j = 0; j < regex.size(); j++)
                                  {

                                      std::smatch matches;

                                      if (std::regex_search(command, matches, std::regex(regex[j])))
                                      {

                                          //Regex matches --> call execute function
                                          dpp::voiceconn *v = event.from->get_voice(event.msg.guild_id);
                                          std::map<dpp::snowflake, dpp::voicestate> voiceMembers;
                                          bool inVoiceChannel = v && v->voiceclient && v->voiceclient->is_ready();

                                          if (inVoiceChannel)
                                          {
                                              voiceMembers = bot.channel_get_sync(v->voiceclient->channel_id).get_voice_members();
                                          }

                                          if ((inVoiceChannel && commands[i]->userMustBeInBotsChannel() && voiceMembers.find(event.msg.author.id) != voiceMembers.end()) || (inVoiceChannel && !commands[i]->userMustBeInBotsChannel()) || (!inVoiceChannel && !commands[i]->userMustBeInBotsChannel()))
                                          {
											std::string match1 = std::string(matches[1]);
											std::string match2 = std::string(matches[2]);
                                              //Bot not in voice channel and command doesn't require user to be in same voice channel
                                              //OR
                                              //Bot in voice channel and command doesn't require user to be in same voice channel
                                              //OR
                                              //Bot in voice and user is in same voice channel and command requires user to be in same voice channel

                                            if(commands[i].get() != &playCommand || isPlayCommandRunning(event.msg.guild_id, playCommandFutures)) {
                                                //Bot already playing --> add command to low priority queue
                                                lowPriorityQueue.push([&commands, event, match1, match2, i] {
                                                    commands[i]->execute(event, match1, match2);
                                                });
                                            } else {
                                                //Bot not playing --> start new thread and start playing
                                                std::packaged_task<void()> packagedTask([&commands, event, match1, match2, i] {
                                                    commands[i]->execute(event, match1, match2);
                                                });
                                                playCommandFutures[event.msg.guild_id] = packagedTask.get_future();

                                                std::thread thread(std::move(packagedTask));
                                                thread.detach();
                                            }
                                          }
                                          else
                                          {
                                              sendErrorEmbed(bot, event, "Please join the same voice channel as the bot to use this command");
                                          }

                                          return;
                                      }
                                  }
                              }

                              //Shortcut commands
                              std::vector<ShortcutEntity> shortcuts = DbUtil::getShortcuts(event.msg.guild_id);
                              for (ShortcutEntity const& shortcut : shortcuts)
                              {
                                std::smatch matches;
                                std::vector<std::string> regex;

                                if(shortcut.command.find(';') != std::string::npos) {
                                    //Multiple regexes/commands
                                    regex = GeneralUtil::explode(shortcut.command, ';');
                                } else {
                                    //Only one regex/command
                                    regex.push_back(shortcut.command);
                                }

                                for (int j = 0; j < regex.size(); j++)
                                {

                                    if (std::regex_search(command, matches, std::regex("^-" + regex[j] + "$")))
                                    {
                                        //Regex matches --> call execute function
                                        dpp::voiceconn *v = event.from->get_voice(event.msg.guild_id);
                                        std::map<dpp::snowflake, dpp::voicestate> voiceMembers;
                                        bool inVoiceChannel = v && v->voiceclient && v->voiceclient->is_ready();

                                        if(inVoiceChannel) {
                                            voiceMembers = bot.channel_get_sync(v->voiceclient->channel_id).get_voice_members();
                                        }

                                        //Check: bot in voice and user is in same voice channel and command requires user to be in same voice channel
                                        if((inVoiceChannel && voiceMembers.find(event.msg.author.id) != voiceMembers.end()) || !inVoiceChannel) {
                                            if(isPlayCommandRunning(event.msg.guild_id, playCommandFutures)) {
                                                //Bot already playing --> add command to low priority queue
                                                lowPriorityQueue.push([&playCommand, event, shortcut] {
                                                    playCommand.execute(event, shortcut.query, std::string());
                                                });
                                            } else {
                                                //Bot not playing --> start new thread and start playing
                                                std::packaged_task<void()> packagedTask([&playCommand, event, shortcut] {
                                                    playCommand.execute(event, shortcut.query, std::string());
                                                });
                                                playCommandFutures[event.msg.guild_id] = packagedTask.get_future();

                                                std::thread thread(std::move(packagedTask));
                                                thread.detach();
                                            }
                                        } else {
                                            sendErrorEmbed(bot, event, "Please join the same voice channel as the bot to use this command");
                                        }

                                        return;
                                    }
                                }
                              } });

    bot.start(false);

    return 0;
}

bool isPlayCommandRunning(dpp::snowflake guildId, std::map<dpp::snowflake, std::shared_future<void>> playCommandFutures)
{
    if (playCommandFutures.find(guildId) == playCommandFutures.end())
    {
        return false;
    }

    std::shared_future<void> future = playCommandFutures.at(guildId);
    std::future_status status = future.wait_for(std::chrono::milliseconds(0));

    //Status should not be finished
    return status != std::future_status::ready;
}

void sendErrorEmbed(dpp::cluster &bot, dpp::message_create_t event, std::string title, std::string description, std::string mediaUri)
{
    sendEmbed(bot, 0xff0000, event, title, description, mediaUri);
}

void sendEmbed(dpp::cluster &bot, uint32_t color, dpp::message_create_t event, std::string title, std::string description, std::string mediaUri)
{
    dpp::embed embed = dpp::embed().set_color(color).set_title(title);

    if (description != std::string())
    {
        embed.set_description(description);
    }

    if (mediaUri != std::string())
    {
        embed.set_image(mediaUri);
    }

    //Reply with the created embed
    bot.message_create(dpp::message(event.msg.channel_id, embed));
}
