#include "shortcutadd.h"
#include "../entities/shortcut.h"
#include "../utils/db.h"
#include "../utils/general.h"

std::vector<std::string> ShortcutAddCommand::getRegex()
{
    std::vector<std::string> regex;
    regex.push_back("^-shortcutadd (.*?) (.*)$");
    regex.push_back("^-sca (.*?) (.*)$");
    return regex;
}

std::string ShortcutAddCommand::getHelp()
{
    return "**-shortcutadd [command] [search term|url], -sca [command] [search term|url]:** Add shortcuts for songs you need often. Separate commands with \";\" to add multiple aliases for the same search term/url. Aliases cannot contain spaces. E.g. \"-sca loutr;l.o.u.t.r https://www.youtube.com/watch?v=z4bEAOeTNsg\"";
}

void ShortcutAddCommand::execute(dpp::message_create_t event, std::string match1, std::string match2)
{
    ShortcutEntity shortcut = ShortcutEntity(match1, match2, event.msg.guild_id);

    //Check if shortcut or alias already exists
    std::vector<std::string> shortcutCommands = GeneralUtil::explode(match1, ';');
    for(const std::string& shortcutCommand : shortcutCommands)
    {
        ShortcutEntity existingShortcut = DbUtil::getShortcutByCommand(shortcutCommand, event.msg.guild_id);
        if (existingShortcut.command != "")
        {
            //Shortcut already exists --> show error message
            error(event, "Error adding shortcut", "The shortcut or alias **" + shortcutCommand + "** already exists.");
            return;
        }
    }

    //Save shortcut to DB
    if (DbUtil::saveShortcut(shortcut))
    {
        success(event, "Shortcut added", "Shortcut(s) **" + match1 + "** added successfully.");
    }
    else
    {
        error(event, "Error adding shortcut");
    }
}
