#include <dpp/nlohmann/json.hpp>
#include <functional>
#include <future>
#include <iostream>
#include <map>
#include <memory>
#include <ogg/ogg.h>
#include <opus/opusfile.h>
#include <ratio>
#include <set>
#include <thread>
#include <vector>
#include <chrono>

#include "audioresampler.h"
#include "av.h"
#include "avutils.h"
#include "codec.h"
#include "ffmpeg.h"
#include "packet.h"
#include "videorescaler.h"

//API2
#include "codec.h"
#include "codeccontext.h"
#include "format.h"
#include "formatcontext.h"

#include "../entities/queue.h"
#include "../utils/config.h"
#include "../utils/db.h"
#include "../utils/general.h"

#include "play.h"

std::vector<std::string> PlayCommand::getRegex()
{
    std::vector<std::string> regex;
    regex.push_back("^-play (.*)$");
    regex.push_back("^-p (.*)$");
    return regex;
}

bool PlayCommand::userMustBeInBotsChannel()
{
    return false;
}

std::string PlayCommand::getHelp()
{
    return "**-play [search term|url], -p [search term|url]:** Adds the given url to the queue or searches for the search term on YouTube and adds the song to the queue. If currently no song is playing the bot will join the voice channel and play the song";
}

void PlayCommand::execute(dpp::message_create_t event, std::string match1, std::string match2)
{
    //Get voice connection
    dpp::voiceconn *v = event.from->get_voice(event.msg.guild_id);
    std::string uri = match1;

    //Add item to queue and send success response
    bool inVoiceChannel = v && v->voiceclient && v->voiceclient->is_ready();
    bool currentlyPlaying = inVoiceChannel && v->voiceclient->is_playing();

    QueueEntity queue = DbUtil::getQueue(event.msg.guild_id);
    QueueItemEntity queueItem = addQueueItem(event, uri, queue, -1, true);

    if (queueItem.id == -1)
    {
        //No result found
        return;
    }

    if (!inVoiceChannel)
    {
        //Currently not in voice channel --> join it
        dpp::guild *g = dpp::find_guild(event.msg.guild_id);
        if (!g->connect_member_voice(event.msg.author.id, false, true))
        {
            error(event, "You don't seem to be in a voice channel! :(");
            return;
        }

        //Register on_voice_ready event to then directly play audio after joining the voice channel
        std::promise<bool> promise;
        std::shared_future<bool> future = promise.get_future();
        dpp::event_handle eventId = bot.on_voice_ready([&promise, event](const dpp::voice_ready_t &voiceEvent) {

            dpp::discord_voice_client *v = voiceEvent.voice_client;

            if(v && v->server_id == event.msg.guild_id && v->is_ready()) {
                promise.set_value(true);
            }
        });

        //Wait until bot joined voice channel
        std::future_status status;
        do
        {
            status = future.wait_for(std::chrono::milliseconds(500));
        } while (status != std::future_status::ready);

        //Remove event since it is not needed anymore
        bot.on_voice_ready.detach(eventId);

        play(event.from->get_voice(event.msg.guild_id)->voiceclient, event, queueItem);
    }
    else if (!currentlyPlaying)
    {
        //Directly play queue item
        int playResult = play(v->voiceclient, event, queueItem);
        if (playResult == 1)
        {
            //Error playing audio (e.g. cannot open input)
            error(event, "Error playing " + queueItem.getDisplayName());
        }
    }
}

QueueItemEntity PlayCommand::addQueueItem(dpp::message_create_t event, std::string uri, QueueEntity queue, int position, bool sendResponse)
{
    bool isUrl = true;
    std::string searchString = uri;

    if (!GeneralUtil::isUrl(uri))
    {
        //Append ytsearch since value isn't a url
        isUrl = false;
        searchString = "ytsearch:" + uri;
    }

    //Try to find search result in database
    SearchEntity search = DbUtil::getSearch(searchString);
    std::string songJson = search.result;
    bool saveSearch = false;
    if (songJson == std::string())
    {
        std::string spotifyClientId = ConfigUtil::get("spotify_client_id");
        std::string spotifyClientSecret = ConfigUtil::get("spotify_client_secret");

        if (std::regex_match(uri, std::regex(R"(^https?:\/\/(open|play)\.spotify\.com\/.+)")) && !spotifyClientId.empty() && !spotifyClientSecret.empty())
        {
            //Normalize Spotify URL
            std::string rewritten_url = std::regex_replace(uri, std::regex(R"(^https?:\/\/(open|play)\.spotify\.com\/)"), "https://open.spotify.com/");
            //Call spotify_dl
            songJson = executeScript(
                "SPOTIPY_CLIENT_ID=\"" + spotifyClientId + "\" " +
                "SPOTIPY_CLIENT_SECRET=\"" + spotifyClientSecret + "\" " +
                "spotify_dl --dump-json -l \"" + rewritten_url + "\""
            );
        }
        else
        {
            //Youtube search or direct link --> execute yt-dlp
            songJson = executeScript("yt-dlp --dump-json \"" + searchString + "\"");
        }

        if (songJson.empty())
        {
            //No result found
            std::string message = "No result found for ";
            if (isUrl)
            {
                message += "[" + uri + "](" + uri + ")";
            }
            else
            {
                message += uri;
            }
            error(event, message);
            return QueueItemEntity();
        }

        search = SearchEntity(searchString, songJson);
        saveSearch = true;
    }

    auto metadata = nlohmann::json::parse(songJson);

    if (saveSearch)
    {
        //Only save search if JSON could be parsed
        DbUtil::saveSearch(search);
    }

    if (metadata.is_array())
    {
        //TODO: foreach over each item (needed for playlists)
        metadata = metadata[0];
    }

    int duration = -1;
    try
    {
        duration = metadata.at("duration");
    }
    catch (nlohmann::detail::out_of_range)
    {
    }
    std::string displayName = metadata.at("fulltitle");
    std::string extractor = metadata.at("extractor");
    std::string videoId = metadata.at("id");

    //Youtube-dl found a youtube video --> create url from id and use it as source
    if (!videoId.empty() && extractor == "youtube")
    {
        uri = "https://www.youtube.com/watch?v=" + videoId;
    }

    std::clog << "Extractor: " << extractor << std::endl;
    // If it is a direct link, extract info via ff
    if (extractor == "generic")
    {
            av::init();
            // av::setFFmpegLoggingLevel(AV_LOG_TRACE);
            av::AudioDecoderContext audioDecoder;
            av::Stream audioStream;
            std::error_code errorCode;
            av::FormatContext inputContext;

            inputContext.openInput(uri, errorCode);
            if (errorCode)
            {
                std::cerr << "Can't open input" << std::endl;
                //return 1;
            }

            //Get first audio stream
            inputContext.findStreamInfo();

            //Extract info from FormatContext
            std::clog << "Generic duration: " << (int)inputContext.duration().seconds() << std::endl;
            duration = inputContext.duration().seconds();
    }

    QueueItemEntity queueItem(queue.id, uri, (displayName.empty() ? uri : displayName), position, duration);

    //Save queue item to db
    std::vector<QueueItemEntity> queueItems = DbUtil::getQueueItems(queue.id);

    if (position == -1)
    {
        //Set highest position as position for queue item
        queueItem.position = queueItems.size();
        queueItems.clear();
        queueItems.push_back(queueItem);
    }
    else
    {
        //Recalculate position of all queue items
        int counter = 0;
        for (int i = 0; i < queueItems.size(); i++)
        {
            if (counter == position)
            {
                counter++;
            }

            queueItems[i].position = counter;
            counter++;
        }

        queueItems.push_back(queueItem);
    }

    //Save queue items to database
    DbUtil::saveQueueItems(queue.id, queueItems);

    //Refetch queue items so that we have the id of the inserted queue item
    queueItems = DbUtil::getQueueItems(queue.id);
    queueItem = queueItems[queueItem.position];

    if (sendResponse)
    {
        success(event, "", "Added " + queueItem.getDisplayName() + " to queue");
    }

    return queueItem;
}

std::string PlayCommand::executeScript(std::string command)
{
    std::string cmd = command;
    std::array<char, 128> buffer;
    std::string result;
    std::unique_ptr<FILE, decltype(&pclose)> pipe(popen(cmd.c_str(), "r"), pclose);
    if (!pipe)
    {
        throw std::runtime_error("popen() failed!");
    }
    while (fgets(buffer.data(), buffer.size(), pipe.get()) != nullptr)
    {
        result += buffer.data();
    }

    return result;
}

int PlayCommand::playNext(dpp::discord_voice_client *v, dpp::message_create_t event, QueueItemEntity queueItem)
{
    QueueItemEntity nextQueueItem;
    std::vector<QueueItemEntity> queueItems = DbUtil::getQueueItems(queueItem.queueId);
    for (int i = 0; i < queueItems.size(); i++)
    {
        if (queueItems[i].id == queueItem.id && i < (queueItems.size() - 1))
        {
            //Next queue item found
            nextQueueItem = queueItems[i + 1];
            break;
        }
    }

    if (nextQueueItem.id == -1)
    {
        if (!autoLeaveThreadStarted)
        {
            startAutoLeaveThread(v, event);
        }
        else
        {
            autoLeaveThread.request_stop();
            startAutoLeaveThread(v, event);
        }

        return 1;
    }

    if (play(v, event, nextQueueItem) == 1)
    {
        //Error playing next queue item
        error(event, "Error playing " + nextQueueItem.getDisplayName());
        return 1;
    }

    return 0;
}

int PlayCommand::play(dpp::discord_voice_client *voiceclient, dpp::message_create_t event, QueueItemEntity queueItem)
{
    // Stop auto leave thread if running
    if (autoLeaveThreadStarted)
    {
        autoLeaveThread.request_stop();
    }

    av::init();
    // av::setFFmpegLoggingLevel(AV_LOG_TRACE);

    int audioStreamIndex = -1;
    av::AudioDecoderContext audioDecoder;
    av::Stream audioStream;
    std::error_code errorCode;

    int count = 0;

    //Open input uri
    av::FormatContext inputContext;
    std::string uri = executeScript("yt-dlp --get-url -f bestaudio/best \"" + queueItem.source + "\"");
    uri.erase(std::remove(uri.begin(), uri.end(), '\n'), uri.end());

    inputContext.openInput(uri, errorCode);
    if (errorCode)
    {
        std::cerr << "Can't open input" << std::endl;
        return 1;
    }

    //Get first audio stream
    inputContext.findStreamInfo();
    for (size_t i = 0; i < inputContext.streamsCount(); ++i)
    {
        auto stream = inputContext.stream(i);
        if (stream.isAudio())
        {
            audioStreamIndex = i;
            audioStream = stream;
            break;
        }
    }

    //std::clog << "Audio stream index: " << audioStreamIndex << std::endl;

    if (audioStream.isNull())
    {
        std::cerr << "Audio stream not found" << std::endl;
        return 1;
    }

    //Check if we can decode the codec
    if (audioStream.isValid())
    {
        audioDecoder = av::AudioDecoderContext(audioStream);
        audioDecoder.open(errorCode);

        if (errorCode)
        {
            std::cerr << "Can't open codec" << std::endl;
            return 1;
        }
    }

    //Output
    av::OutputFormat outputFormat;
    av::FormatContext outputContext;

    //Set output codec
    outputFormat.setFormat("opus", "opus");
    outputContext.setFormat(outputFormat);
    std::clog << "Output format: " << outputFormat.name() << " / " << outputFormat.longName() << std::endl;

    //Create audio encoder
    av::Codec outputCodec = av::findEncodingCodec(outputFormat, false);
    av::AudioEncoderContext audioEncoder(outputCodec);

    std::clog << outputCodec.name() << " / " << outputCodec.longName() << ", audio: " << (outputCodec.type() == AVMEDIA_TYPE_AUDIO) << std::endl;

    //Settings for audioDecoder
    auto sampleFormats = outputCodec.supportedSampleFormats();
    audioEncoder.setSampleRate(48000);
    audioEncoder.setSampleFormat(sampleFormats[0]);
    audioEncoder.setChannels(2);
    audioEncoder.setChannelLayout(AV_CH_LAYOUT_STEREO);
    audioEncoder.setTimeBase(av::Rational(1, audioEncoder.sampleRate()));

    //Open audio encoder
    audioEncoder.open(errorCode);
    if (errorCode)
    {
        std::cerr << "Can't open encoder" << std::endl;
        return 1;
    }

    std::clog << "Encoder frame size: " << audioEncoder.frameSize() << std::endl;

    //Resample audio
    av::AudioResampler resampler(audioEncoder.channelLayout(), audioEncoder.sampleRate(), audioEncoder.sampleFormat(), audioDecoder.channelLayout(), audioDecoder.sampleRate(), audioDecoder.sampleFormat());

    //Send success response, audio is probably playing correctly + set current queue item to currently playing song
    DbUtil::setCurrentQueueItem(queueItem.queueId, queueItem.id);
    success(event, "", "Playing #" + std::to_string(queueItem.rowNumber) + ": " + queueItem.getDisplayName());

    //Process input file
    int i = 0;
    while (true)
    {
        av::Packet packet = inputContext.readPacket(errorCode);
        if (errorCode)
        {
            std::clog << "Packet reading error: " << errorCode << ", " << errorCode.message() << std::endl;
            break;
        }

        if (packet.streamIndex() != audioStreamIndex)
        {
            //Not reading from correct audio stream in input file
            continue;
        }

        // std::clog << "Read packet: isNull=" << (bool)!packet << ", " << packet.pts() << "(nopts:" << packet.pts().isNoPts() << ")"
        // << " / " << packet.pts().seconds() << " / " << packet.timeBase() << " / st: " << packet.streamIndex() << std::endl;

        av::AudioSamples samples = audioDecoder.decode(packet, errorCode);
        count++;

        if (errorCode)
        {
            //Ffmpeg error --> continue to next song
            std::cerr << "Decode error: " << errorCode << ", " << errorCode.message() << std::endl;
        }
        else if (!samples)
        {
            std::cerr << "Empty samples set" << std::endl;

            //if (!pkt) //decoder flushed here
            //   break;
            //continue;
        }

        //Empty samples set should not be pushed to the resampler, but it is valid case for the
        //end of reading: during samples empty, some cached data can be stored at the resampler
        //internal buffer, so we should consume it.
        if (samples)
        {
            resampler.push(samples, errorCode);
            if (errorCode)
            {
                std::clog << "Resampler push error: " << errorCode << ", text: " << errorCode.message() << std::endl;
                continue;
            }
        }

        //Pop resampler data
        bool getAll = !samples;
        while (true)
        {
            av::AudioSamples outputSamples(audioEncoder.sampleFormat(),
                                           audioEncoder.frameSize(),
                                           audioEncoder.channelLayout(),
                                           audioEncoder.sampleRate());

            //Resample
            bool hasFrame = resampler.pop(outputSamples, getAll, errorCode);
            if (errorCode)
            {
                std::clog << "Resampling status: " << errorCode << ", text: " << errorCode.message() << std::endl;
                break;
            }
            else if (!hasFrame)
            {
                break;
            }

            //Encode audio
            outputSamples.setStreamIndex(0);
            outputSamples.setTimeBase(audioEncoder.timeBase());

            av::Packet outputPacket = audioEncoder.encode(outputSamples, errorCode);
            if (errorCode)
            {
                std::cerr << "Encoding error: " << errorCode << ", " << errorCode.message() << std::endl;
                return 1;
            }
            else if (!outputPacket)
            {
                //cerr << "Empty packet\n";
                continue;
            }

            outputPacket.setStreamIndex(0);

            // std::clog << "Write packet: pts=" << outputPacket.pts() << ", dts=" << outputPacket.dts() << " / " << outputPacket.pts().seconds() << " / " << outputPacket.timeBase() << " / st: " << outputPacket.streamIndex() << std::endl;

            //Send packet to Discord if encode isn't aborted
            int guildIdEncodeAborted = isGuildIdEncodeAborted(event.msg.guild_id);
            if (guildIdEncodeAborted != 0)
            {
                //Flush buffer of discord bot and ffmpeg
                voiceclient->stop_audio();
                inputContext.close();
                outputContext.flush();
                outputContext.close();

                if (guildIdEncodeAborted == 2)
                {
                    //Leave voice channel
                    event.from->disconnect_voice(event.msg.guild_id);
                }
                return 2;
            }

            //Pause encoding if bot is paused
            if (voiceclient->is_paused())
            {
                while (voiceclient->is_paused())
                {
		    std::this_thread::sleep_for(std::chrono::seconds(1));
                }
            }

            voiceclient->send_audio_opus(outputPacket.data(), outputPacket.size());
        }

        //For the first packets samples can be empty: decoder caching
        if (!packet && !samples)
        {
            break;
        }

        i++;
    }

    //Is sampler flushed?
    std::cerr << "Delay: " << resampler.delay() << std::endl;

    //Flush encoder queue
    std::clog << "Flush encoder:" << std::endl;
    while (true)
    {
        av::AudioSamples null(nullptr);
        av::Packet outputPacket = audioEncoder.encode(null, errorCode);
        if (errorCode || !outputPacket)
        {
            break;
        }

        outputPacket.setStreamIndex(0);

        //std::clog << "Write packet: pts=" << outputPacket.pts() << ", dts=" << outputPacket.dts() << " / " << outputPacket.pts().seconds() << " / " << outputPacket.timeBase() << " / st: " << outputPacket.streamIndex() << std::endl;

        //Send packet to Discord if encode isn't aborted
        int guildIdEncodeAborted = isGuildIdEncodeAborted(event.msg.guild_id);
        if (guildIdEncodeAborted != 0)
        {
            //Flush buffer of discord bot
            voiceclient->stop_audio();
            inputContext.close();
            outputContext.flush();
            outputContext.close();

            if (guildIdEncodeAborted == 2)
            {
                //Leave voice channel
                event.from->disconnect_voice(event.msg.guild_id);
            }

            return 2;
        }

        //Pause encoding if bot is paused
        if (voiceclient->is_paused())
        {
            while (voiceclient->is_paused())
            {
	        std::this_thread::sleep_for(std::chrono::seconds(1));
            }
        }

        voiceclient->send_audio_opus(outputPacket.data(), outputPacket.size());
    }

    inputContext.close();
    outputContext.flush();
    outputContext.close();

    //Wait until voice client finished playing the current song
    if (voiceclient->is_playing())
    {
        bool isPlaying = true;
        do
        {

            int guildIdEncodeAborted = isGuildIdEncodeAborted(event.msg.guild_id);
            if (guildIdEncodeAborted != 0)
            {
                //Flush buffer of discord bot
                voiceclient->stop_audio();

                if (guildIdEncodeAborted == 2)
                {
                    //Leave voice channel
                    event.from->disconnect_voice(event.msg.guild_id);
                }

                return 2;
            }

            try
            {
                isPlaying = voiceclient->is_playing();
            }
            catch (...)
            {
                //Sometimes there are mutex lock exceptions
                isPlaying = true;
            }

	    std::this_thread::sleep_for(std::chrono::seconds(1));
        } while (isPlaying);
    }

    //Play the next item
    playNext(voiceclient, event, queueItem);

    return 0;
}

int PlayCommand::isGuildIdEncodeAborted(dpp::snowflake guildId)
{

    if (guildIdAbortedEncodes.find(guildId) == guildIdAbortedEncodes.end())
    {
        //Encode not aborted
        return 0;
    }

    int encodeAborted = guildIdAbortedEncodes[guildId];
    if (encodeAborted != 0)
    {
        //Encode aborted --> set it to false so that the next encode won't abort
        guildIdAbortedEncodes[guildId] = 0;
    }

    return encodeAborted;
}

void PlayCommand::startAutoLeaveThread(dpp::discord_voice_client *v, dpp::message_create_t event)
{
    autoLeaveThread = std::jthread([v, event](std::stop_token stopToken) {

        int sleepIntervalInSeconds = 2;
        int totalSleepTime = 0;

        while(!stopToken.stop_requested() && totalSleepTime < 3600) { //Sleep for 1h or until stop
            std::this_thread::sleep_for(std::chrono::seconds(sleepIntervalInSeconds));
            totalSleepTime += sleepIntervalInSeconds;
        }

        //Stop not requested --> wait 1h --> leave voice channel
        if (!stopToken.stop_requested() && v && v->is_ready() && !v->is_playing() && !v->is_paused()) {
            event.from->disconnect_voice(event.msg.guild_id);
        }
    });

    autoLeaveThread.detach();
    autoLeaveThreadStarted = true;
}